<?php

class coreFunctions
{

    static function getCoreSitePath()
    {
        if (!defined("_CONFIG_CORE_SITE_HOST_URL"))
        {
            return WEB_ROOT;
        }

        return _CONFIG_SITE_PROTOCOL . "://" . _CONFIG_CORE_SITE_FULL_URL;
    }

    static function mainSiteAccessOnly()
    {
        // make sure this is the main site, only uploads on the main site are permitted
        if (self::currentIsMainSite() == false)
        {
            self::redirect(_CONFIG_SITE_PROTOCOL . '://' . _CONFIG_CORE_SITE_HOST_URL);
        }
    }

    static function currentIsMainSite()
    {
        if (_CONFIG_SITE_HOST_URL != _CONFIG_CORE_SITE_HOST_URL)
        {
            return false;
        }

        return true;
    }

    // converts a date/timestamp into the specified format
    static function formatDate($date = null, $format = null)
    {
        if (is_null($format))
        {
            if (defined('SITE_CONFIG_DATE_TIME_FORMAT'))
            {
                $format = SITE_CONFIG_DATE_TIME_FORMAT;
            }
            else
            {
                $format = 'Y-m-d H:i:s';
            }
        }

        if (is_null($date))
        {
            return;
        }

        if ($date == '0000-00-00 00:00:00')
        {
            return;
        }

        // if $date contains only numbers, treat it as a timestamp
        if (ctype_digit($date) === true)
            return date($format, $date);
        else
            return date($format, strtotime($date));
    }

    // redirects user to $url
    static function redirect($url = null)
    {
        // if no url assume redirect to self
        if (is_null($url))
        {
            $url = $_SERVER['PHP_SELF'];
        }

        // if no headers sent
        if (!headers_sent())
        {
            header('location: ' . $url);
            exit;
        }

        // fallback to meta/javascript redirect
        echo '<script type="text/javascript">';
        echo 'window.location.href="' . $url . '";';
        echo '</script>';
        echo '<noscript>';
        echo '<meta http-equiv="refresh" content="0;url=' . $url . '" />';
        echo '</noscript>';
        exit;
    }

    // fixes MAGIC_QUOTES
    static function fixSlashes($arr = '')
    {
        if (is_null($arr) || $arr == '')
        {
            return null;
        }

        if (!get_magic_quotes_gpc())
        {
            return $arr;
        }

        return is_array($arr) ? array_map('coreFunctions::fixSlashes', $arr) : stripslashes($arr);
    }

    static function convertToBytes($val)
    {
        $val  = trim($val);
        $last = strtolower($val[strlen($val) - 1]);
        switch ($last)
        {
            // The 'G' modifier is available since PHP 5.1.0
            case 'g':
                $val *= 1024;
            case 'm':
                $val *= 1024;
            case 'k':
                $val *= 1024;
        }

        return $val;
    }

    static function getPHPMaxUpload()
    {
        $postMaxSize       = self::convertToBytes(ini_get('post_max_size'));
        $uploadMaxFilesize = self::convertToBytes(ini_get('upload_max_filesize'));
        if ($postMaxSize > $uploadMaxFilesize)
        {
            return $uploadMaxFilesize;
        }

        return $postMaxSize;
    }

    // grabs the contents of a remote URL. Can perform basic authentication if un/pw are provided.
    static function getRemoteUrlContent($url, $username = null, $password = null)
    {
        if (function_exists('curl_init'))
        {
            $ch   = curl_init();
            if (!is_null($username) && !is_null($password))
                curl_setopt($ch, CURLOPT_HTTPHEADER, array('Authorization: Basic ' . base64_encode("$username:$password")));
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
            $html = curl_exec($ch);
            curl_close($ch);
            return $html;
        }
        elseif (ini_get('allow_url_fopen') == true)
        {
            if (!is_null($username) && !is_null($password))
                $url  = str_replace("://", "://$username:$password@", $url);
            $html = file_get_contents($url);
            return $html;
        }
        else
        {
            // Cannot open url. Either install curl-php or set allow_url_fopen = true in php.ini
            return false;
        }
    }

    // returns the current date time in sql format
    static function sqlDateTime()
    {
        return date("Y-m-d H:i:s");
    }

    static function getDirectoryListing($path)
    {
        if (substr($path, strlen($path) - 1, 1) != '/')
        {
            $path .= '/';
        }

        $items  = array();
        if ($handle = opendir($path))
        {
            while (false !== ($entry = readdir($handle)))
            {
                if ($entry != "." && $entry != "..")
                {
                    $items[] = $path . $entry;
                }
            }
            closedir($handle);
        }

        return $items;
    }

    static function convertDateToTimestamp($date, $format = 'Y-m-d H:i:s')
    {
        if (!validation::validDate($date, $format))
        {
            return false;
        }

        $d = DateTime::createFromFormat($format, $date);

        return $d->getTimestamp();
    }

    // get users IP address, cloudflare & proxy nginx friendly
    static function getUsersIPAddress()
    {
        // cloudflare
        if ((isset($_SERVER['HTTP_CF_CONNECTING_IP'])) && strlen($_SERVER['HTTP_CF_CONNECTING_IP']))
        {
            return $_SERVER['HTTP_CF_CONNECTING_IP'];
        }

        // for nginx proxy to apache users
        if ((isset($_SERVER['HTTP_X_REAL_IP'])) && strlen($_SERVER['HTTP_X_REAL_IP']))
        {
            return $_SERVER['HTTP_X_REAL_IP'];
        }

        return $_SERVER['REMOTE_ADDR'];
    }

    static function dateformatPhpToJqueryUi($php_format)
    {
        $SYMBOLS_MATCHING = array(
            // Day
            'd' => 'dd',
            'D' => 'D',
            'j' => 'd',
            'l' => 'DD',
            'N' => '',
            'S' => '',
            'w' => '',
            'z' => 'o',
            // Week
            'W' => '',
            // Month
            'F' => 'MM',
            'm' => 'mm',
            'M' => 'M',
            'n' => 'm',
            't' => '',
            // Year
            'L' => '',
            'o' => '',
            'Y' => 'yy',
            'y' => 'y',
            // Time
            'a' => '',
            'A' => '',
            'B' => '',
            'g' => '',
            'G' => '',
            'h' => '',
            'H' => '',
            'i' => '',
            's' => '',
            'u' => ''
        );
        $jqueryui_format  = "";
        $escaping         = false;
        for ($i = 0; $i < strlen($php_format); $i++)
        {
            $char = $php_format[$i];
            if ($char === '\\') // PHP date format escaping character
            {
                $i++;
                if ($escaping)
                    $jqueryui_format .= $php_format[$i];
                else
                    $jqueryui_format .= '\'' . $php_format[$i];
                $escaping = true;
            }
            else
            {
                if ($escaping)
                {
                    $jqueryui_format .= "'";
                    $escaping = false;
                }
                if (isset($SYMBOLS_MATCHING[$char]))
                    $jqueryui_format .= $SYMBOLS_MATCHING[$char];
                else
                    $jqueryui_format .= $char;
            }
        }
        return $jqueryui_format;
    }

    static function output404()
    {
        header("HTTP/1.0 404 Not Found");
        exit;
    }

    static function outputFailureImage()
    {
        $localFailureImage = DOC_ROOT . "/themes/" . SITE_CONFIG_SITE_THEME . "/images/trans_1x1.gif";
        header('Content-type: image/gif');
        echo file_get_contents($localFailureImage);
        die();
    }

    /**
     * Format bytes into human readable
     * 
     * @param integer $bytes
     * @param string $return
     * @return string
     */
    static function formatSize($bytes, $return = 'both', $topRounding = true)
    {
        $size = $bytes / 1024;
        $ext  = 'KB';
        if ($size < 1024)
        {
            if ($topRounding == true)
            {
                $size = number_format($size, 2);
            }
            else
            {
                $size = substr(number_format($size, 3), 0, 4);
            }
            $ext = 'KB';
        }
        else
        {
            if ($size / 1024 < 1024)
            {
                if ($topRounding == true)
                {
                    $size = number_format($size / 1024, 2);
                }
                else
                {
                    $size = substr(number_format($size / 1024, 3), 0, 4);
                }
                $ext = 'MB';
            }
            else if ($size / 1024 / 1024 < 1024)
            {
                if ($topRounding == true)
                {
                    $size = number_format($size / 1024 / 1024, 2);
                }
                else
                {
                    $size = substr(number_format($size / 1024 / 1024, 3), 0, 4);
                }
                $ext = 'GB';
            }
            else if ($size / 1024 / 1024 / 1024 < 1024)
            {
                if ($topRounding == true)
                {
                    $size = number_format($size / 1024 / 1024 / 1024, 2);
                }
                else
                {
                    $size = substr(number_format($size / 1024 / 1024 / 1024, 3), 0, 4);
                }
                $ext = 'TB';
            }
        }
        // remove unneccessary zeros
        $size = str_replace(".00 ", " ", $size);

        switch ($return)
        {
            case 'size':
                return $size;
            case 'ext':
                return $ext;
            default:
                return $size . ' ' . $ext;
        }
    }

    static function ensureSiteMatchesConfig()
    {
		// ignore if cli mode
		if((defined('CLI_MODE')) && (CLI_MODE == true))
		{
			return;
		}
		
        // get requesting host
        $siteHost = self::getHttpHost();

        // remove any ports
        $siteHostExp   = explode(':', $siteHost);
        $configHostExp = explode(':', _CONFIG_SITE_HOST_URL);

        // redirect to config file version if not
        if (strtolower($siteHostExp[0]) != strtolower($configHostExp[0]))
        {
            self::redirect(_CONFIG_SITE_PROTOCOL . '://' . _CONFIG_SITE_FULL_URL);
        }
    }

    static function getHttpHost()
    {
        return $_SERVER['HTTP_HOST'];
    }

    static function flushOutput()
    {
        //ob_end_flush();
        ob_flush();
        flush();
        //ob_start();
    }

    static function calculateDownloadSpeedFormatted($filesize, $speed = 0)
    {
        if ($speed == 0)
        {
            // assume 5MB as an average
            $speed = 5242880;
        }

        $minutes = ceil($filesize / $speed);

        return self::secsToHumanReadable($minutes);
    }

    static function secsToHumanReadable($secs)
    {
        $units = array(
            "week"   => 7 * 24 * 3600,
            "day"    => 24 * 3600,
            "hour"   => 3600,
            "minute" => 60,
            "second" => 1,
        );

        // specifically handle zero
        if ($secs == 0)
            return "0 " . t('seconds', 'seconds');

        $s = "";

        foreach ($units as $name => $divisor)
        {
            if ($quot = intval($secs / $divisor))
            {
                $label = t($name, $name);
                if (abs($quot) > 1)
                {
                    $label = t($name . 's', $name . 's');
                }
                $s .= "$quot $label";
                $s .= " ";
                $secs -= $quot * $divisor;
            }
        }

        return substr($s, 0, -1);
    }

    static function getTmpFolderSize($formatted = false)
    {
        $bytes = disk_total_space(sys_get_temp_dir());
        if ($bytes == 0)
        {
            return 0;
        }

        if ($formatted == true)
        {
            return coreFunctions::formatSize($bytes);
        }

        return $bytes;
    }

    static function decideShowMaintenancePage()
    {
        if ((!defined('IGNORE_MAINTENANCE_MODE')) && (!isset($_REQUEST['csaKey1'])))
        {
            // Initialize current user
            $Auth = Auth::getAuth();

            // check for maintenance mode
            if ((SITE_CONFIG_MAINTENANCE_MODE == 'yes') && ($Auth->level_id <= 2))
            {
                self::showMaintenancePage();
            }
        }
    }

    static function showMaintenancePage()
    {
        if (!defined('IGNORE_MAINTENANCE_MODE'))
        {
            include_once(CORE_ROOT . '/page/_maintenance_page.inc.php');
            exit;
        }
    }

    // sends an HTML formatted email
    function sendHtmlEmail($to, $subject, $msg, $fromEmail = null, $plaintext = '', $debug = false, $fromName = null, $useTemplate = true)
    {
        // include email class
        require_once CORE_ROOT . '/includes/email_class/class.phpmailer.php';

        if (!is_array($to))
        {
            $to = array($to);
        }

        if ($fromEmail == null)
        {
            $fromEmail = SITE_CONFIG_DEFAULT_EMAIL_ADDRESS_FROM;
        }

        if ($fromName == null)
        {
            $fromName = SITE_CONFIG_SITE_NAME;
        }

        $headerStyling = '';
        $footerStyling = '';
        if (($useTemplate == true) && (SITE_CONFIG_EMAIL_TEMPLATE_ENABLED == 'yes'))
        {
            // email header
            if (strlen(SITE_CONFIG_EMAIL_TEMPLATE_HEADER))
            {
                $headerStyling = SITE_CONFIG_EMAIL_TEMPLATE_HEADER;
            }

            // email footer
            if (strlen(SITE_CONFIG_EMAIL_TEMPLATE_FOOTER))
            {
                $footerStyling = SITE_CONFIG_EMAIL_TEMPLATE_FOOTER;
            }
        }

        if (strlen($headerStyling) == 0)
        {
            $headerStyling = '<style type="text/css">';
            $headerStyling .= 'body { font: 11px Verdana,Geneva,Arial,Helvetica,sans-serif; }\n';
            $headerStyling .= '</style>';
        }
        
        // do template replacements
        $replacements = array();
        $replacements['SITE_NAME'] = SITE_CONFIG_SITE_NAME;
        $replacements['WEB_ROOT'] = WEB_ROOT;
        $replacements['DATE_NOW'] = date(SITE_CONFIG_DATE_FORMAT);
        $replacements['DATE_TIME_NOW'] = date(SITE_CONFIG_DATE_TIME_FORMAT);
        $replacements['SITE_IMAGE_PATH'] = SITE_IMAGE_PATH;
        if(strlen($headerStyling))
        {
            foreach($replacements AS $k=>$v)
            {
                $headerStyling = str_replace('[[['.$k.']]]', $v, $headerStyling);
                $footerStyling = str_replace('[[['.$k.']]]', $v, $footerStyling);
            }
        }

        $msg = $headerStyling . $msg . $footerStyling;

        // send using smtp
        if ((SITE_CONFIG_EMAIL_METHOD == 'smtp') && (strlen(SITE_CONFIG_EMAIL_SMTP_HOST)))
        {
            $error = '';
            $mail  = new PHPMailer();
            $body  = $msg;
            $body  = eregi_replace("[\]", '', $body);

            $mail->IsSMTP();
            try
            {
                $mail->Host      = SITE_CONFIG_EMAIL_SMTP_HOST;
                $mail->SMTPDebug = 1;
                $mail->SMTPAuth  = (SITE_CONFIG_EMAIL_SMTP_REQUIRES_AUTH == 'yes') ? true : false;
                $mail->Host      = SITE_CONFIG_EMAIL_SMTP_HOST;
                $mail->Port      = SITE_CONFIG_EMAIL_SMTP_PORT;
                if (SITE_CONFIG_EMAIL_SMTP_REQUIRES_AUTH == 'yes')
                {
                    $mail->Username = SITE_CONFIG_EMAIL_SMTP_AUTH_USERNAME;
                    $mail->Password = SITE_CONFIG_EMAIL_SMTP_AUTH_PASSWORD;
                }

                $mail->SetFrom($fromEmail, $fromName);
                $mail->AddReplyTo($fromEmail, $fromName);
                $mail->Subject = $subject;

                if (strlen($plaintext))
                {
                    $mail->AltBody = $plaintext; // optional
                }

                $mail->MsgHTML($body);
                foreach ($to as $address)
                {
                    $mail->AddAddress($address);
                }
                $mail->Send();
            }
            catch (phpmailerException $e)
            {
                $error = $e->errorMessage();
            }
            catch (Exception $e)
            {
                $error = $e->getMessage();
            }

            if (strlen($error))
            {
                if ($debug == true)
                {
                    echo $error;
                }
                return false;
            }

            return true;
        }

        // send using php mail
        foreach ($to as $address)
        {
            $boundary = uniqid(rand(), true);

            $headers = "From: $fromEmail\n";
            $headers .= "MIME-Version: 1.0\n";
            $headers .= "Content-Type: multipart/alternative; boundary = $boundary\n";
            $headers .= "This is a MIME encoded message.\n\n";
            $headers .= "--$boundary\n" .
                "Content-Type: text/plain; charset=UTF-8\n" .
                "Content-Transfer-Encoding: base64\n\n";
            $headers .= chunk_split(base64_encode($plaintext));
            $headers .= "--$boundary\n" .
                "Content-Type: text/html; charset=UTF-8\n" .
                "Content-Transfer-Encoding: base64\n\n";
            $headers .= chunk_split(base64_encode($msg));
            $headers .= "--$boundary--\n" .
                mail($address, $subject, '', $headers);
        }
    }

    static function getReffererDomainOnly()
    {
        // the referer isn't always available
        if (!isset($_SERVER['HTTP_REFERER']))
        {
            return false;
        }

        $fullRefererUrl = strtolower(trim($_SERVER['HTTP_REFERER']));
        $urlData        = parse_url($fullRefererUrl);
        $host           = $urlData['host'];
        if (isset($urlData['port']))
        {
            if (($urlData['port'] != 80) && ((int) $urlData['port'] != 0))
            {
                $host .= ':' . $urlData['port'];
            }
        }

        return $host;
    }

    static function allowCrossSiteAjax()
    {
        // for cross domain uploads
        $refDomain = self::getReffererDomainOnly();
        if (!$refDomain)
        {
            $refDomain = _CONFIG_SITE_PROTOCOL . '://' . _CONFIG_CORE_SITE_HOST_URL;
        }
        else
        {
            $refDomain = _CONFIG_SITE_PROTOCOL . "://" . str_replace(array("http://", "https://"), "", $refDomain);
        }

        header('Access-Control-Allow-Origin: ' . (($refDomain === false) ? WEB_ROOT : ($refDomain)));
        header('Access-Control-Allow-Headers: Content-Type, Content-Range, Content-Disposition, Content-Description');
        header('Access-Control-Allow-Credentials: true');
    }

    static function outputCaptcha()
    {
        // include the captcha functions
        require_once CORE_ROOT . '/includes/recaptcha/recaptchalib.php';

        // return the captcha html
        return recaptcha_get_html(SITE_CONFIG_CAPTCHA_PUBLIC_KEY);
    }

    static function captchaCheck($challengeField, $responseField)
    {
        // include the captcha functions
        require_once CORE_ROOT . '/includes/recaptcha/recaptchalib.php';

        // check captcha
        $resp = recaptcha_check_answer(SITE_CONFIG_CAPTCHA_PRIVATE_KEY, $_SERVER["REMOTE_ADDR"], $challengeField, $responseField);
        if (!$resp->is_valid)
        {
            return false;
        }

        return true;
    }

    static function createUploadError($name, $msg)
    {
        // setup object for errors
        $fileUpload                    = new stdClass();
        $fileUpload->size              = 0;
        $fileUpload->type              = '';
        $fileUpload->name              = $name;
        $fileUpload->error             = $msg;
        $fileUpload->error_result_html = uploader::generateErrorHtml($fileUpload);

        return json_encode(array($fileUpload));
    }

    static function timeToRelativeString($ts)
    {
        if (!ctype_digit($ts))
        {
            $ts = strtotime($ts);
        }

        $diff = time() - $ts;
        if ($diff == 0)
        {
            return 'now';
        }
        elseif ($diff > 0)
        {
            $day_diff = floor($diff / 86400);
            if ($day_diff == 0)
            {
                if ($diff < 60)
                    return 'just now';
                if ($diff < 120)
                    return '1 minute ago';
                if ($diff < 3600)
                    return floor($diff / 60) . ' minutes ago';
                if ($diff < 7200)
                    return '1 hour ago';
                if ($diff < 86400)
                    return floor($diff / 3600) . ' hours ago';
            }
            if ($day_diff == 1)
                return 'yesterday';
            if ($day_diff < 7)
                return $day_diff . ' days ago';
            if ($day_diff < 31)
                return ceil($day_diff / 7) . ' weeks ago';
            if ($day_diff < 60)
                return 'last month';
            return date('F Y', $ts);
        }
    }
	
	// no longer used, in favour of cron tasks
	static function executeBatchTasks()
	{
	}
    
    // used for remote upload progress cross site
    static function removeSubDomain($host)
    {
        $hostNames = explode(".", $host);
        
        return $hostNames[count($hostNames)-2] . "." . $hostNames[count($hostNames)-1];
    }

}
