//================================================================
//
// leadInOutObject.js   | Object which displays and manages lead in / out data 
//                       on ArtCAM toolpaths
//
//

// ----------------------------------------------------------------------------
// COPYRIGHT 2001 DELCAM PLC., BIRMINGHAM, ENGLAND. 
// ----------------------------------------------------------------------------
//
// History.
// Who When     What   
// --- -------- ---------------------------------------------------------------
// bem 24/05/01 Written
// bem 21/06/01 Added error checking
// pah 05/02/04 Get/set default distance in registry under:
//              "Defaults" | "LeadInOutDistance"
// pah 19/04/04 Added <td> IDs so that fields can be hidden
// pah 24/08/04 Added cutter comp
// pah 01/10/04 Save/load cutter comp as a flag
// pah 08/10/04 Added lead angles
// pah 14/12/04 Use StoreDouble/RetrieveDouble from utils.js to convert units
// pah 17/08/05 Hide cutter comp by default - HTML page must enable it
// pah 12/04/06 Allow zero length leads, but no if cutter comp set
// pah 08/11/06 Added ramped leads option
//-----------------------------------------------------------------------------


//=== LeadInOut Constructor ===================================================
//
// History
// Who When     What
// --- -------- ---------------------------------------------------------------
// bem 24/05/2001 Written
//-----------------------------------------------------------------------------

function LeadInOut
   (
   name,
   changed_callback 
   )
{
   // Assign Page Members
   this.name = name;

   this.useLeadInOut  = false;
   this.distance      = 0.0; 
   this.doCircular    = false;
   this.radius        = 0.0;
   // sap 30/04/02 add support for overcutting
   this.overcut       = 0.0;
   this.leadin_angle  = 45;
   this.leadout_angle = 45;
   this.EditMode      = false;

   // Assign Functions
   this.writeDiv              = LeadInOut_WriteDiv;
   this.initialise            = LeadInOut_Initialise;
   this.checkValid            = LeadInOut_CheckValid;
   this.onToggleLeadInOut     = LeadInOut_OnOff;
   this.toggleHelp            = LeadInOut_ToggleHelp;
   this.onTypeRadioClick      = LeadInOut_OnTypeRadioClick;
   this.setEditingMode        = LeadInOut_SetEditingMode;

   this.updateParametersFromToolpath = LeadInOut_UpdateParametersFromToolpath;
   this.saveParametersToToolpath     = LeadInOut_SaveParametersToToolpath;

   // is there a callback ?
   if (changed_callback == null)
      {
      this.doCallback = false;
      }
   else
      {
      this.updateCallback = changed_callback
      this.doCallback = true;
      }

   // get path to the root of our html - we use this to build
   // addresses to our images
   this.image_dir = artcam.HtmlRootDir();
   this.image_dir += "..\\SharedImages\\LeadInOut\\";

}

//=== LeadInOut_WriteDiv ==============================================
//
// Writes out the html which is the interface for the ramping process into any toolpath dialog
//
// History
// Who When     What
// --- -------- ---------------------------------------------------------
// bem 24/05/01 Written
// pah 04/04/04 Entry specific help comment
//-----------------------------------------------------------------------------

function LeadInOut_WriteDiv()
{
   // This is a bit like create. We write all our document objects into the page
   // then show and hide them as we wish.

   var basic_profiling = false;
   if (!artcam.IsFeatureLicensed(FT_2DMACH_CUTTERCOMP) && !artcam.IsFeatureLicensed(FT_2DMACH_RAMPEDLEADS))
      basic_profiling = true;

/*
   document.writeln("  <tr><td></td><td height=2px></td></tr>");
   document.writeln("  <tr><td></td><td bgcolor=buttonshadow></td></tr>");
   document.writeln("  <tr><td></td><td bgcolor=buttonhighlight></td></tr>");
   document.writeln("  <tr><td></td><td height=2px></td></tr>");
*/

   // sap 09/05/02 add an id to the first check
   document.writeln("  <tr id='" + this.name + "DoLeadTableDiv' >");
   document.writeln("    <td valign=top><img src='"+ this.image_dir + "LeadInOut.gif'></td>");
   document.writeln("    <td>");
   document.writeln("    <nobr>");
   document.writeln("       <INPUT type='checkbox' id='" + this.name + "LeadInOutCheck' onclick='javascript:" + this.name + ".onToggleLeadInOut()'>");
   document.writeln("       <span><b>" + jslLeadInOutHeader + "&nbsp;&nbsp;</b></span>");
   document.writeln("    </nobr>");
   document.writeln("    </td>");
   document.writeln("  </tr>");


   document.writeln("   <tr>");
   document.writeln("     <td></td>");
   document.writeln("     <td colspan=2 id=" + this.name + "LeadInOutTableDiv style='display:none'>");
   document.writeln("       <table border=0 cellspacing=0>");
   document.writeln("           <tr id='" + this.name + "NoLeadOutCheckRow' >");
   document.writeln("             <td colspan=2><b>" + jslNoLeadOut + ":</b>&nbsp;");
   document.writeln("                <input id='" + this.name + "NoLeadOutCheck' type=checkbox ></td>");
   document.writeln("           </tr>");
   document.writeln("           <tr>");
   document.writeln("             <td colspan=2><b>" + jslLeadInOutDistance + ":</b>&nbsp;");
   document.writeln("                <input id='" + this.name + "LeadInOutDistanceEdit' type=text size=5 maxLength=8 value='0' number></td>");
   document.writeln("           </tr>");
   // sap 30/04/02 add support for overcutting
   document.writeln("           <tr id='" + this.name + "LeadInOutOverCutRow'>");
   document.writeln("             <td colspan=2><b>" + jslLeadInOutOverCut + ":</b>&nbsp;");
   document.writeln("                <input id='" + this.name + "LeadInOutOverCutEdit' type=text size=5 maxLength=8 value='0' number></td>");
   document.writeln("           </tr>");

   document.writeln("           <tr>");
   document.writeln("             <td>");
   document.writeln("                 <img src='"+ this.image_dir + "LeadInOutLinear.gif' border=0>");
   document.writeln("             </td>");
   document.writeln("             <td>");
   document.writeln("               <input name='" + this.name + "jLeadInOutTypeRadio' onclick='" + this.name + ".onTypeRadioClick()' type=radio CHECKED ></input>");
   document.writeln("                &nbsp;<b>" + jslLeadInOutLinear + "</b>");
   document.writeln("             </td>");
   document.writeln("           </tr>");

   document.writeln("           <tr id='" + this.name + "AngleInRow'>");
   document.writeln("             <td></td>");
   document.writeln("             <td><span id='" + this.name + "LeadInAngleLabel'><b>" + jslLeadInAngle + ":</b><span>");
   document.writeln("                <input id='" + this.name + "LeadInAngleEdit' type=text size=5 maxLength=8 value='0' number></td>");
   document.writeln("           </tr>");
   document.writeln("           <tr id='" + this.name + "AngleOutRow'>");
   document.writeln("             <td></td>");
   document.writeln("             <td><span id='" + this.name + "LeadOutAngleLabel'><b>" + jslLeadOutAngle + ":</b><span>");
   document.writeln("                <input id='" + this.name + "LeadOutAngleEdit' type=text size=5 maxLength=8 value='0' number></td>");
   document.writeln("           </tr>");


   document.writeln("           <tr id='" + this.name + "RampedLeadsDiv'>");
   document.writeln("             <td></td>");
   document.writeln("             <td>"); 
   document.writeln("               <input id='" + this.name + "RampedLeadsCheck' type=checkbox unchecked >");
   document.writeln("               <span id='" + this.name + "RampedLeadsLabel'>&nbsp;&nbsp;<b>" + jslRampedLeads + "</b>&nbsp;</span>");
   document.writeln("             </td>");
   document.writeln("           </tr>");


   document.writeln("           <tr>");
   document.writeln("             <td>");
   document.writeln("                 <img src='"+ this.image_dir + "LeadInOutCircular.gif'>");
   document.writeln("             </td>");
   document.writeln("             <td>");
   document.writeln("               <input name='" + this.name + "jLeadInOutTypeRadio' onclick='" + this.name + ".onTypeRadioClick()' type=radio ></input>&nbsp;<b>" + jslLeadInOutCircular + "</b>");
   document.writeln("             </td>");
   document.writeln("           </tr>");
   document.writeln("           <tr>");
   document.writeln("             <td></td>");
   document.writeln("             <td><span id='" + this.name + "LeadInOutRadiusLabel'>&nbsp;&nbsp;<b>" + jslLeadInOutRadius + ":</b>&nbsp;</span>");
   document.writeln("                 <input id='" + this.name + "LeadInOutRadiusEdit' type=text size=5 maxLength=8 value='0' number>");
   document.writeln("             </td>");
   document.writeln("           </tr>");
   // sap 09/05/02 add an id to the auto place check
   document.writeln("           <tr id='" + this.name + "LeadInOutPositionTableDiv'>");
   document.writeln("            <td colspan=2>"); 
   document.writeln("               <span id='" + this.name + "LeadInOutPositionLabel'>&nbsp;&nbsp;<b>" + jslLeadInOutPosition + ":</b>&nbsp;</span>");
   document.writeln("               <input id='" + this.name + "LeadInOutPositionCheck' type=checkbox checked >");
   document.writeln("             </td>");
   document.writeln("           </tr>");

   document.writeln("           <tr id='" + this.name + "LeadInOutCutterCompDiv'>");
   document.writeln("            <td colspan=2>"); 
   document.writeln("               <span id='" + this.name + "LeadInOutCutterCompLabel'>&nbsp;&nbsp;<b>" + jslLeadInOutCutterComp + ":</b>&nbsp;</span>");
   document.writeln("               <input id='" + this.name + "LeadInOutCutterCompCheck' type=checkbox unchecked >");
   document.writeln("             </td>");
   document.writeln("           </tr>");

   document.writeln("       </table>");
   document.writeln("       </td>");
   document.writeln("    </tr>");
   document.writeln("    <tr HELPTAG style='display:none'>");
   if (basic_profiling)
      document.writeln("      <td></td><td id='" + this.name + "rowLeadInOutHelp' colspan=2>" + jslLeadInOutEntryHelpText);
   else
      document.writeln("      <td></td><td id='" + this.name + "rowLeadInOutHelp' colspan=2>" + jslLeadInOutHelpText);
   document.writeln("      </td>");
   document.writeln("    </tr>");

}



//=== LeadInOut_Initialise ==============================================
//
// History
// DICC  Who When     What
// ----- --- -------- ---------------------------------------------------------
// 00000 sap 02/05/01 Written
//-----------------------------------------------------------------------------

function LeadInOut_Initialise()
{
   // Load default lead distance from registry
   this.distance = RetrieveDoubleDimension("Defaults", "LeadInOutDistance", 0.0 );

   // Create members attached to the objects in our Div on the page.
   // This is done by evaluating the name that we called it when we wrote it in writeDiv
   // Since the object has this global name the evaluation returns the object
   // which we store in our object for convenience.
   // A bit like do data exchange in CDialog!

   // link up the divs that display the data etc
   
   // Set up members for the data controls
   this.LeadInOutCheck        = eval( this.name + 'LeadInOutCheck');
   this.LeadInOutTableDiv	   = eval( this.name + 'LeadInOutTableDiv');
   this.LeadInOutDistanceEdit = eval( this.name + 'LeadInOutDistanceEdit');

   this.LeadInAngleEdit           = eval( this.name + 'LeadInAngleEdit');
   this.LeadOutAngleEdit          = eval( this.name + 'LeadOutAngleEdit');
   this.LeadInAngleLabel      = eval( this.name + 'LeadInAngleLabel');
   this.LeadOutAngleLabel     = eval( this.name + 'LeadOutAngleLabel');

   this.LeadInOutTypeRadio    = eval( this.name + 'jLeadInOutTypeRadio');
   this.LeadInOutRadiusLabel  = eval( this.name + 'LeadInOutRadiusLabel');
   this.LeadInOutRadiusEdit   = eval( this.name + 'LeadInOutRadiusEdit');
   this.LeadInOutPosCheck     = eval( this.name + 'LeadInOutPositionCheck');
   this.HelpDiv               = eval( this.name + 'rowLeadInOutHelp'); 
   this.NoLeadOutCheck        = eval( this.name + 'NoLeadOutCheck');
   this.LeadInOutOverCutEdit = eval( this.name + 'LeadInOutOverCutEdit');
   this.LeadInOutCutterCompCheck = eval( this.name + 'LeadInOutCutterCompCheck');
   this.RampedLeadsCheck     = eval( this.name + 'RampedLeadsCheck');

   this.DoLeadTableDiv            = eval( this.name + 'DoLeadTableDiv' );
   this.LeadInOutPositionTableDiv = eval( this.name + 'LeadInOutPositionTableDiv' );
   this.LeadInOutCutterCompDiv = eval( this.name + 'LeadInOutCutterCompDiv' );
   this.RampedLeadsDiv       = eval( this.name + 'RampedLeadsDiv' );
   this.RampedLeadsLabel     = eval( this.name + 'RampedLeadsLabel');
   this.AngleInRow           = eval( this.name + 'AngleInRow' );
   this.AngleOutRow          = eval( this.name + 'AngleOutRow' );

   // work out which ones are to be displayed and which hidden
   this.LeadInOutTableDiv.style.display  = "none";

   // Hide cutter comp, ramped by default - HTML page must enable it
   this.LeadInOutCutterCompDiv.style.display  = "none";
   this.RampedLeadsDiv.style.display  = "none";

   this.LeadInOutDistanceEdit.value = this.distance;
   this.LeadInAngleEdit.value = this.leadin_angle;
   this.LeadOutAngleEdit.value = this.leadout_angle;

   this.onTypeRadioClick();
} 


//=== LeadInOut_OnOff ==============================================
//
// History
// Who When     What
// --- -------- ---------------------------------------------------------
// sap 03/05/01 Written
// pah 20/04/04 Made callback work
//-----------------------------------------------------------------------------

function LeadInOut_OnOff()
{
   // work out which ones are to be displayed and which hidden
   if ( this.LeadInOutCheck.checked == true )
      this.LeadInOutTableDiv.style.display  = "";
   else
      this.LeadInOutTableDiv.style.display  = "none";

   if (this.doCallback)
      this.updateCallback();
}


// === LeadInOut_UpdateParametersFromToolpath ===================================
//
// Update ramp parameters from passed toolpath
//
// History
// Who When     What
// --- -------- ---------------------------------------------------------------
// bem 21/05/01 Written
// pah 14/12/04 Support model units on dimensioned values
// ------------------------------------------------------------------------------
function LeadInOut_UpdateParametersFromToolpath(toolpath)
{
   if (toolpath == null)
      return;

   this.LeadInOutCheck.checked = toolpath.GetFlag('DoLeadInOut',this.LeadInOutCheck.checked);
   this.LeadInOutDistanceEdit.value = GetToolpathDimension( toolpath, 'LeadInOutDistance',parseFloat( this.LeadInOutDistanceEdit.value)); 

   var doCircular   = toolpath.GetFlag('DoCircularLeadInOut',this.doCircular);
   this.LeadInOutRadiusEdit.value = GetToolpathDimension( toolpath, 'LeadInOutRadius',parseFloat( this.LeadInOutRadiusEdit.value ));
   
   var do_auto = toolpath.GetFlag('AutoPositionLeadInOut',this.LeadInOutPosCheck.checked);
   this.LeadInOutOverCutEdit.value = GetToolpathDimension( toolpath, 'LeadInOutoverCut', parseFloat( this.LeadInOutOverCutEdit.value ));
   
   // Cutter compensation
   this.LeadInOutCutterCompCheck.checked = toolpath.GetFlag('CutterCompensation', this.LeadInOutCutterCompCheck.checked);

   // Ramped leads
   this.RampedLeadsCheck.checked = toolpath.GetFlag('RampedLeads', this.RampedLeadsCheck.checked);

   // Angles
   this.LeadInAngleEdit.value = toolpath.GetDouble('LeadInAngle', parseFloat( this.LeadInAngleEdit.value ));
   this.LeadOutAngleEdit.value = toolpath.GetDouble('LeadOutAngle', parseFloat( this.LeadOutAngleEdit.value ));

   // Update display   
   if ( doCircular )
      this.LeadInOutTypeRadio[1].checked = true;
   else 
      this.LeadInOutTypeRadio[0].checked = true;
    
   if ( !do_auto )
      this.LeadInOutPosCheck.checked = false;
   
   if (this.LeadInOutCheck.checked || this.EditMode)
      this.LeadInOutTableDiv.style.display  = "";
   else 
      this.LeadInOutTableDiv.style.display  = "none";

   this.onTypeRadioClick();
}

// === LeadInOut_SaveParametersToToolpath ===================================
//
// Save ramp parameters to passed toolpath
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 21/05/2001 Written
// ----------------------------------------------------------------------

function LeadInOut_SaveParametersToToolpath(toolpath)
{
   if (toolpath == null)
      return;

   // Read user choice of distance and angles
   var dist = parseFloat( this.LeadInOutDistanceEdit.value);
   var lin_angle = parseFloat( this.LeadInAngleEdit.value);
   var lout_angle = parseFloat( this.LeadOutAngleEdit.value);

   toolpath.SetFlag('DoLeadInOut',this.LeadInOutCheck.checked);
   SetToolpathDimension( toolpath, 'LeadInOutDistance', dist );

   toolpath.SetFlag('DoCircularLeadInOut',this.LeadInOutTypeRadio[1].checked); 
   SetToolpathDimension( toolpath, 'LeadInOutRadius', parseFloat( this.LeadInOutRadiusEdit.value ));

   toolpath.SetFlag('AutoPositionLeadInOut',this.LeadInOutPosCheck.checked);
   SetToolpathDimension( toolpath, 'LeadInOutoverCut', parseFloat( this.LeadInOutOverCutEdit.value ));

   // Angles
   toolpath.SetDouble('LeadInAngle', lin_angle);
   toolpath.SetDouble('LeadOutAngle', lout_angle);

   // Cutter compensation
   toolpath.SetFlag('CutterCompensation', this.LeadInOutCutterCompCheck.checked);

   // Ramped leads
   toolpath.SetFlag('RampedLeads', this.RampedLeadsCheck.checked);

   // Also set default distance in registry
   StoreDoubleDimension("Defaults", "LeadInOutDistance", dist );
}

// === LeadInOut_ToggleHelp =======================================
//
// Toggle display of our help
//

function LeadInOut_ToggleHelp()
{
   ToggleVisibility(this.HelpDiv);
}

// === LeadInOut_OnTypeRadioClick =======================================
//
// Called when user toggles use of circular arc lead ins
//
function LeadInOut_OnTypeRadioClick()
{
   this.doCircular = this.LeadInOutTypeRadio[1].checked;

   var circle = false;
   if (this.doCircular)
      circle = true;

   this.LeadInOutRadiusLabel.style.color= !circle ? 'graytext' : 'windowtext';
   this.LeadInOutRadiusEdit.disabled = !circle;
   this.LeadInOutRadiusEdit.readOnly = !circle;	
   this.LeadInOutRadiusEdit.style.backgroundColor= !circle ? 'buttonface' : 'window';

   this.LeadOutAngleLabel.style.color= circle ? 'graytext' : 'windowtext';
   this.LeadOutAngleEdit.disabled = circle;
   this.LeadOutAngleEdit.readOnly = circle;	
   this.LeadOutAngleEdit.style.backgroundColor= circle ? 'buttonface' : 'window';

   this.LeadInAngleLabel.style.color= circle ? 'graytext' : 'windowtext';
   this.LeadInAngleEdit.disabled = circle;
   this.LeadInAngleEdit.readOnly = circle;	
   this.LeadInAngleEdit.style.backgroundColor= circle ? 'buttonface' : 'window';

   // Disable ramped leads and cutter comp if edit mode or circular leads
   var  dis = (circle || this.EditMode)
   this.RampedLeadsLabel.style.color= dis ? 'graytext' : 'windowtext';
   this.RampedLeadsCheck.disabled = dis;
   this.RampedLeadsCheck.readOnly = dis;	

   this.LeadInOutCutterCompCheck.style.color= dis ? 'graytext' : 'windowtext';
   this.LeadInOutCutterCompCheck.disabled = dis;
   this.LeadInOutCutterCompCheck.readOnly = dis;	

}


// === LeadInOut_CheckValid ======================================
//
// Check values for lead in a re valid - returns error message if not
//

function LeadInOut_CheckValid()
{
  var error = "";
  // if not doing lead in / out no possible error with values
  if (this.LeadInOutCheck.checked == false)
     return error;

  var comp = this.LeadInOutCutterCompCheck.checked;
  var distance = parseFloat(this.LeadInOutDistanceEdit.value);

  // If cutter comp lead cannot be zero
  if (distance <= 0.0 && comp)
     error += jslLeadInOutZeroDistanceError;

  var overcut = parseFloat(this.LeadInOutDistanceEdit.value);
  if (overcut < 0.0)
     error += jslLeadInOutZeroOverCutError;


  // if doing circular lead in distance must be <= radius
  if (this.LeadInOutTypeRadio[1].checked)
     {
     var radius = parseFloat(this.LeadInOutRadiusEdit.value);

     if (radius <= 0.0 > radius)
        error += jslLeadInOutZeroRadiusError ;

     if (distance > radius)
        error += jslLeadInOutZeroDistanceGRadiusError ;
     }
 
 return error;
}


//=== LeadInOut_SetEditingMode ==============================================
//
// History
// Who When     What
// --- -------- ---------------------------------------------------------
// sap 09/05/02 Written
//-----------------------------------------------------------------------

function LeadInOut_SetEditingMode()
{
   // show main body
   this.LeadInOutTableDiv.style.display  = "";
   this.EditMode = true;
   
   // hide the do lead row
   this.DoLeadTableDiv.style.display  = "none";
   // hide the auto place leads row
   this.LeadInOutPositionTableDiv.style.display  = "none";
}
